/**
* \file: daemon_fsm.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>

#include "control/daemon_fsm.h"

#include "model/daemon_model.h"
#include "model/challenge.h"
#include "control/levelchanger.h"
#include "control/ald.h"
#include "control/configuration.h"
#include "util/logger.h"
#include "util/helper.h"

typedef enum daemon_fsm_recovery_state_t {
	NO_RECOVERY_ONGOING,
	STARTUP_RECOVERY_ONGOING,
	RECOVERY_AFTER_TIMEOUT_ONGOING
} daemon_fsm_recovery_state_t;

static daemon_fsm_recovery_state_t recovery_state = NO_RECOVERY_ONGOING;

//---------------------------------------- private functions --------------------------------------------------------
static level_change_request_callback_t request_callback=NULL;

static void daemon_fsm_signal_handle_recovery_on_script_exec_tmout(bool stay_with_current_levels,
								   error_code_t result);
static void daemon_fsm_enter_recovery_state(bool startup_recovery);
static void daemon_fsm_enter_level_replay_state(void);
static error_code_t daemon_fsm_prepare_kickoff_recovery(security_level_t recovery_level);
static void daemon_fsm_handle_level_change_done(bool stay_with_current_levels,
						bool level_completed_with_errors,
						error_code_t result);
static void daemon_fsm_enter_idle_state();
static void daemon_fsm_enter_level_change_ongoing_state(const security_level_t targeted_level);
static void daemon_fsm_immediate_shutdown(void);
static void daemon_fsm_schedule_shutdown(daemon_level_state_t level_state);
static bool daemon_fsm_check_replay_trigger(void);
static void daemon_fsm_create_level_change_complete_file(void);
static bool daemon_fsm_check_level_change_complete_file_exits(void);
static void daemon_fsm_remove_level_change_complete_file(void);
static void daemon_fsm_remove_replay_trigger_file_if_exists(void);
//---------------------------------------- API members --------------------------------------------------------------
void daemon_fsm_kickoff(void)
{
	daemon_level_state_t level_state = daemon_model_get_level_state();

	if (level_state != DAEMON_LEVEL_INITIALIZATION)
	{
		logger_log_error("DAEMON_FSM - Implementation error in daemon detected. Trying to kick off the daemon fsm"
				" in level state %s / service state %s.",
				daemon_model_get_level_state_name(level_state),
				daemon_model_get_service_state_name(daemon_model_get_service_state()));
		return;
	}
	if (daemon_model_is_recovery_needed())
	{
		// detected an incomplete permanent level change
		// or the need to close the device due to a corrupted state file
		daemon_fsm_enter_recovery_state(true);
	}
	else if (daemon_fsm_check_replay_trigger())
	{
		daemon_fsm_enter_level_replay_state();
	}
	else
	{
		if (!daemon_fsm_check_level_change_complete_file_exits()) {
			/*
			 * currently this situation will happen directly after initialization
			 * deployed ALD state file has identical target and current levels but
			 * no level change done so far
			 */
			logger_log_info("DAEMON_FSM - file that marks level change completion is missing although no change pending - create it.");
			daemon_fsm_create_level_change_complete_file();
		}

		/* notification about current level */
		app_iface_signal_level_changed(daemon_model_get_active_level());

		daemon_fsm_enter_idle_state();
	}
}

/*
 * daemon_fsm_signal_level_change_complete is called to evaluate the result of
 * level change. It will determine if the sequence is completed or a
 * recovery action needs to be performed
 *
 * Overall level-change/lock/replay recovery sequence:
 * In case of success it will stop at the corresponding item.
 * In case of errors(needing recovery to continue) it will proceed with the next item in the list.
 *  - level change, lock or replay of current level
 *  - Try error recovery to last known persistent level (same power cycle)
 *  - Wait till next startup - retry error recovery to last known persistent level
 *  - Try recovery to fail-safe-level 0 (same power cycle)
 *  - Wait till next startup - retry recovery to fail-safe-level 0
 *
 *  If all this fails the recovery logic will give up and mark that the level
 *  change has been completed with error (LEVEL_CHANGE_COMPLETED_WITH_ERRORS).
 *  The final level can be either 0 or last known persistent level.
 *  Which value is used depends on the kind of error observed.
 *
 */
void daemon_fsm_signal_level_change_complete(error_code_t result)
{
	daemon_level_state_t level_state;
	bool stay_with_current_levels=false;
	bool recovery_needed=false;

	//check state we are coming from
	level_state=daemon_model_get_level_state();
	if ((level_state == DAEMON_LEVEL_INITIALIZATION) || (level_state == DAEMON_LEVEL_STABLE))
	{
		logger_log_error("DAEMON_FSM - Got a \'change complete\' event in a state where we are not expecting it."
				" Ignoring it.");
		return;
	}

	/* Set default for recovery_needed based on recovery_state
	 * For normal level changes we accept some scripts failing with error or skipped
	 * but for recovery we don't accept anything like that */
	if (recovery_state != NO_RECOVERY_ONGOING)
		recovery_needed=true;

	if (result == RESULT_SCRIPTS_FAILED)
	{
		logger_log_info("Level change completed partly successfully. One or more of the scripts returned"
				" a non zero exit code.");
		logger_log_errmem("ALD - Level change completed partly successfully. One or more of the scripts returned"
				" a non zero exit code.");
	}
	else if (result == RESULT_SCRIPTS_MODIFIED)
	{
		logger_log_info("Level change completed partly successfully. One or more of the scripts have been"
				" modified. The level change has been completed nonetheless.");
		logger_log_errmem("ALD - Level change completed partly successfully. One or more of the scripts have been"
				" modified. The level change has been completed nonetheless.");
	}
	else if (result == RESULT_SCRIPT_EXEC_TIMEOUT)
	{
		/* we don't know if scripts of 1, 2, 3 ... or none features have been executed before */
		logger_log_info("The level change request timed out.");
		logger_log_errmem("ALD - The level change request timed out.");
		recovery_needed=true; /* always recover */
		stay_with_current_levels=true;
	}
	else if (result != RESULT_OK)
	{
		logger_log_info("The level change failed with error code %d. Staying with the current levels.", (int)result);
		logger_log_errmem("ALD - The level change failed with error code %d. Staying with the current levels.", (int)result);
		stay_with_current_levels=true;
	}
	else
	{
		logger_log_info("Level change completed successfully.");
		logger_log_errmem("ALD - Level change completed successfully.");
		recovery_needed = false; /* everything worked -- recovery not needed */
	}

	if(recovery_needed != true)
	{
		/* The level change is marked as complete if
		 * 	- the level change or recovery was either successful or
		 *        partially successful
		 *      - we are staying with the current level due to errors */
		daemon_fsm_handle_level_change_done(stay_with_current_levels,
						    (result != RESULT_OK), /* completed with errors */
						    result);
	}
	else
	{
		/* In case of timeout or in case of script execution errors in recovery
		 * we might need to try recovery (or recovery on startup) */
		daemon_fsm_signal_handle_recovery_on_script_exec_tmout(stay_with_current_levels, result);
	}
}

void daemon_fsm_signal_level_change_request(const challenge_response_t *response,
		level_change_request_callback_t callback)
{
	daemon_level_state_t level_state = daemon_model_get_level_state();
	error_code_t result=RESULT_OK;
	char tmp_string[RESPONSE_SERIAL_NUMBER_SIZE]={0};

	// normally, we are expecting a '\0' terminated string. The following is just to prevent problems in case the given
	// string does not fullfill the expectation
	strncpy(tmp_string, response->serial_number, RESPONSE_SERIAL_NUMBER_SIZE-1);
	logger_log_debug("User with serial number \'%s\' requested a level change from level %d to level %d.",
					tmp_string,daemon_model_get_active_level(),response->targeted_level);

	logger_log_info("Client ALD protocol version:%d.%d",response->client_ALD_protocol_major,response->client_ALD_protocol_minor);

	//check state
	if (level_state == DAEMON_LEVEL_CHANGE_ONGOING)
		result=RESULT_DAEMON_BUSY_LEVEL_CHANGE;
	else if (level_state == DAEMON_LEVEL_RECOVERY_ONGOING)
		result=RESULT_DAEMON_BUSY_LEVEL_RECOVERY;
	/* don't accept any level exceeding max level */
	else if (response->targeted_level > MAX_LEVEL)
		result=RESULT_INVALID_LEVEL;

	//check challenge response
	if (result==RESULT_OK)
		result=challenge_validate_response(response);

	//kick of the request
	if (result==RESULT_OK)
	{
		challenge_invalidate();
		//store the callback so that the FSM can signal the app iface about the finished request once it is done
		request_callback=callback;
		daemon_fsm_enter_level_change_ongoing_state(response->targeted_level);
	}
	else
	{
		if (result==RESULT_DAEMON_BUSY_LEVEL_CHANGE)
		{
			logger_log_error("Unable to process the incoming request. A level change is currently ongoing.");
			logger_log_errmem("ALD - Unable to process the incoming request. A level change is currently ongoing.");
		}
		else if (result==RESULT_DAEMON_BUSY_LEVEL_RECOVERY)
		{
			logger_log_error("Unable to process the incoming request. A level recovery is currently ongoing.");
			logger_log_errmem("ALD - Unable to process the incoming request. A level recovery is currently ongoing.");
		}
		else if (result==RESULT_INVALID_LEVEL)
		{
			logger_log_error("Unable to process the incoming request. No configuration found for requested level %d or max level exceeded.",
					response->targeted_level);
			logger_log_errmem("ALD - Unable to process the incoming request. No configuration found for requested level %d or max level exceeded.",
					response->targeted_level);
		}
		else if (result==RESULT_CHALLENGE_EXPIRED)
		{
			logger_log_error("Unable to process the incoming request. The challenge expired.");
			logger_log_errmem("ALD - Unable to process the incoming request. The challenge expired.");
		}
		else
		{
			logger_log_error("Unable to process the incoming request. The challenge is not solved correctly.");
			logger_log_errmem("ALD - Unable to process the incoming request. The challenge is not solved correctly.");
		}

		//inform the app iface about the failed request
		if (callback != NULL)
			callback(result, false);
	}
}

void daemon_fsm_signal_lock_device_request(level_change_request_callback_t callback)
{
	daemon_level_state_t level_state = daemon_model_get_level_state();
	error_code_t result=RESULT_OK;

	//check state
	if (level_state == DAEMON_LEVEL_CHANGE_ONGOING)
		result=RESULT_DAEMON_BUSY_LEVEL_CHANGE;
	else if (level_state == DAEMON_LEVEL_RECOVERY_ONGOING)
		result=RESULT_DAEMON_BUSY_LEVEL_RECOVERY;

	//kick of the request
	if (result==RESULT_OK)
	{
		//store the callback so that the FSM can signal the app iface about the finished request once it is done
		request_callback=callback;
		daemon_fsm_enter_level_change_ongoing_state(0);
	}
	else
	{
		if (result==RESULT_DAEMON_BUSY_LEVEL_CHANGE)
		{
			logger_log_error("Unable to process the incoming request. A level change is currently ongoing.");
			logger_log_errmem("ALD - Unable to process the incoming request. A level change is currently ongoing.");
		}
		else if (result==RESULT_DAEMON_BUSY_LEVEL_RECOVERY)
		{
			logger_log_error("Unable to process the incoming request. A level recovery is currently ongoing.");
			logger_log_errmem("ALD - Unable to process the incoming request. A level recovery is currently ongoing.");
		}
		else if (result==RESULT_INVALID_LEVEL)
		{
			logger_log_error("Unable to process the incoming request. No configuration found for level 0.");
			logger_log_errmem("ALD - Unable to process the incoming request. No configuration found for level 0.");
		}
		//inform the app iface about the failed request
		if (callback != NULL)
			callback(result, false);
	}
}

void daemon_fsm_shutdown_request(void)
{
	daemon_level_state_t level_state = daemon_model_get_level_state();

	daemon_fsm_schedule_shutdown(level_state);
}

void daemon_fsm_signal_shutdown_request(void)
{
	daemon_level_state_t level_state = daemon_model_get_level_state();
	daemon_service_state_t service_state = daemon_model_get_service_state();

	logger_log_debug("DAEMON_FSM - Shutdown request received. We are currently in level state %s / service state %s.",
			daemon_model_get_level_state_name(level_state),
			daemon_model_get_service_state_name(service_state));

	if (service_state == DAEMON_SERVICE_SHUTDOWN_PENDING) {
		logger_log_error("The daemon received a second shutdown request while it is still processing a level change."
						" This is taken as request to go down immediately without finishing the level change.");
		daemon_fsm_immediate_shutdown();
	} else {
		daemon_fsm_schedule_shutdown(level_state);
	}
}
//-------------------------------------------------------------------------------------------------

//------------------------------------- private members -------------------------------------------
static void daemon_fsm_signal_handle_recovery_on_script_exec_tmout(bool stay_with_current_levels,
								   error_code_t result)
{
	daemon_service_state_t service_state = daemon_model_get_service_state();

	if (request_callback!=NULL)
	{
		request_callback(result, !stay_with_current_levels);
		request_callback=NULL;
	}

	if ( recovery_state == STARTUP_RECOVERY_ONGOING ) {
		if (daemon_model_get_targeted_persisted_level() == FAIL_SAFE_LEVEL) {
			logger_log_info("DAEMON_FSM - Even startup recovery to fail-safe-level failed - give up.");
			logger_log_errmem("ALD - DAEMON_FSM - Even startup recovery to fail-safe-level failed - give up.");
			daemon_fsm_handle_level_change_done(stay_with_current_levels, true , result);

			/* early abort of the function */
			return;
		}

		logger_log_info("DAEMON_FSM - startup recovery failed - schedule recovery to fail-safe-level.");
		logger_log_errmem("ALD - DAEMON_FSM - startup recovery failed - schedule recovery to fail-safe-level.");
		daemon_model_schedule_fail_safe_recovery();
		if (daemon_model_persist()!=RESULT_OK) {
			logger_log_error("DAEMON_FSM - we didn't manage to store the trigger for recovery to fail-safe-level."
					" This will probably cause the last recovery actions to be repeated in the next startup.");
			logger_log_errmem("ALD - DAEMON_FSM - we didn't manage to store the trigger for recovery to fail-safe-level."
					" This will probably cause the last recovery actions to be repeated in the next startup.");
		}
		/* continue with daemon_fsm_immediate_shutdown or daemon_fsm_enter_recovery_state */
	}


	if (service_state == DAEMON_SERVICE_SHUTDOWN_PENDING)
	{
		logger_log_info("DAEMON_FSM - Daemon shutdown requested."
				" Continue/retry recovery in the next startup.");
		logger_log_errmem("ALD - DAEMON_FSM - Daemon shutdown requested."
				" Continue/retry recovery in the next startup.");
		daemon_fsm_immediate_shutdown();
	} else {
		if (recovery_state == RECOVERY_AFTER_TIMEOUT_ONGOING) {
			logger_log_info("DAEMON_FSM - Recovery timed out/failed."
					" Retry recovery in the next startup.");
			logger_log_errmem("ALD - DAEMON_FSM - Recovery timed out/failed."
					" Retry recovery in the next startup.");

			daemon_model_set_state_idle(true);
		} else {
			/* recovery_state is NO_RECOVERY_ONGOING or STARTUP_RECOVERY_ONGOING */
			daemon_fsm_enter_recovery_state(false);
		}
	}
}

static void daemon_fsm_enter_level_replay_state(void)
{
	security_level_t replay_level;

	replay_level=daemon_model_get_persisted_level();
	logger_log_info("A level replay action is initiated to level %d.",replay_level);
	logger_log_errmem("ALD - A level replay action is initiated to level %d.",replay_level);

	request_callback=NULL;
	daemon_fsm_enter_level_change_ongoing_state(replay_level);
}

static error_code_t daemon_fsm_prepare_kickoff_recovery(security_level_t recovery_level)
{
	error_code_t result;
	change_type_t type;

	app_iface_signal_level_changing(recovery_level);

	daemon_fsm_remove_level_change_complete_file();

	daemon_model_set_state_recovery_ongoing(recovery_level);

	result=levelchanger_prepare_levelchange(recovery_level, NULL);
	if (result==RESULT_OK)
	{
		if (recovery_state == STARTUP_RECOVERY_ONGOING)
		{
			type=ALD_RECOVERY_CHANGE;
		} else {
			type=ALD_RECOVERY_ON_TIMEOUT;
		}
		result=levelchanger_kickoff_levelchange(type);
	} else {
		levelchanger_unprepare_levelchange();
	}
	return result;
}

static void daemon_fsm_enter_recovery_state(bool startup_recovery)
{
	error_code_t result;
	security_level_t recovery_level;

	recovery_level=daemon_model_get_recovery_level();

	logger_log_info("A recovery action is initiated to restore level %d.",recovery_level);
	logger_log_errmem("ALD - A recovery action is initiated to restore level %d.",recovery_level);

	if (startup_recovery == true)
		recovery_state = STARTUP_RECOVERY_ONGOING;
	else
		recovery_state = RECOVERY_AFTER_TIMEOUT_ONGOING;

	result = daemon_fsm_prepare_kickoff_recovery(recovery_level);

	/* if setting up the recovery fails at startup recovery (recovery_mode_initiated != true)
	 * try recovery to fail save level - if not already done before */
	if (result != RESULT_OK)
	{
		logger_log_error("Recovery to restore level %d failed.",recovery_level);
		logger_log_errmem("ALD - Recovery to restore level %d failed.",recovery_level);
		daemon_fsm_signal_level_change_complete(result);
	}
}

static void daemon_fsm_handle_level_change_done(bool stay_with_current_levels,
						bool level_completed_with_errors,
						error_code_t result)
{
	security_level_t active_level;
	security_level_t persisted_level;

	/* set and store levels */
	if (stay_with_current_levels)
	{
		//stay in the model with the current level
		active_level=daemon_model_get_active_level();
		persisted_level=daemon_model_get_persisted_level();
	}
	else
	{
		//change the model to the targeted levels
		active_level=daemon_model_get_targeted_active_level();
		persisted_level=daemon_model_get_targeted_persisted_level();
	}

	daemon_model_set_stable_levels(active_level, persisted_level, level_completed_with_errors);

	if (daemon_model_persist() != RESULT_OK) {
		logger_log_error("DAEMON_FSM - We changed to level %d but we didn't manage to store the state persistently."
			 " That will probably trigger some recovery login in the next startup.",active_level);
		logger_log_errmem("ALD - DAEMON_FSM - We changed to level %d but we didn't manage to store the state persistently."
			 " That will probably trigger some recovery login in the next startup.",active_level);

		if (result == RESULT_OK)
			result = RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES;
	} else {
		/* create level change complete file */
		daemon_fsm_create_level_change_complete_file();
	}

	/* notification about changed level */
	app_iface_signal_level_changed(active_level);

	if (request_callback!=NULL)
	{
		request_callback(result, !stay_with_current_levels);
		request_callback=NULL;
	}

	recovery_state = NO_RECOVERY_ONGOING;

	daemon_fsm_enter_idle_state(stay_with_current_levels, result);
}


static void daemon_fsm_enter_idle_state()
{
	daemon_model_set_state_idle(false);

	logger_log_debug("DAEMON_FSM - Entered state IDLE.");

	if (daemon_model_get_service_state() == DAEMON_SERVICE_SHUTDOWN_PENDING)
		daemon_fsm_immediate_shutdown();
}

static void daemon_fsm_enter_level_change_ongoing_state(const security_level_t targeted_level)
{
	bool is_permanent_level_change=false;
	error_code_t result;

	daemon_fsm_remove_level_change_complete_file();

	result=levelchanger_prepare_levelchange(targeted_level, &is_permanent_level_change);

	if (is_permanent_level_change)
	{
		daemon_model_set_state_change_ongoing(targeted_level, targeted_level);
	}
	else
	{
		// in case we have a volatile level change and we are coming from a permanent level different from 0, we are
		// changing indirectly permanently back to 0, which mean that we in fact have a permanent level change ongoing
		if (daemon_model_get_persisted_level() != 0)
		{
			daemon_model_set_state_change_ongoing(targeted_level,0);
			is_permanent_level_change=true;
		}
		else
		{
			daemon_model_set_state_change_ongoing(targeted_level,daemon_model_get_persisted_level());
		}
	}
	app_iface_signal_level_changing(targeted_level);

	if (result==RESULT_OK)
	{
		if(daemon_model_persist()!=RESULT_OK)
		{
			logger_log_error("Error storing new level persistently. Level change request canceled.");
			levelchanger_unprepare_levelchange();
			daemon_fsm_signal_level_change_complete(RESULT_PERSISTENT_STATE_FILE_ACCESS_ISSUES);
			return;
		}

		logger_log_debug("DAEMON_FSM - Start changing to level %d.",targeted_level);
		result=levelchanger_kickoff_levelchange(ALD_NORMAL_CHANGE);
	}

	if (result!=RESULT_OK)
	{
		levelchanger_unprepare_levelchange();
		daemon_fsm_signal_level_change_complete(result);
	}
}

static void daemon_fsm_immediate_shutdown(void)
{
	logger_log_debug("DAEMON_FSM - Kicking of shutdown.");
	ald_shutdown();
}

static void daemon_fsm_schedule_shutdown(daemon_level_state_t level_state)
{
	switch(level_state)
	{
		case DAEMON_LEVEL_STABLE:
		case DAEMON_LEVEL_RECOVERY_ONSTARTUP_NEEDED:
			/* we can shutdown directly */
			daemon_fsm_immediate_shutdown();
			break;
		case DAEMON_LEVEL_CHANGE_ONGOING:
		case DAEMON_LEVEL_RECOVERY_ONGOING:
			logger_log_debug("DAEMON_FSM - A level change is currently ongoing. The shutdown is delayed until the"
					" level change is finished.");
			daemon_model_set_state_shutting_down();
			break;
		default:
			/*Do Nothing*/
			break;
	}
}

static bool daemon_fsm_check_replay_trigger(void)
{
	struct stat stat_result;
	return stat(configuration_get_replay_level_trigger_path(), &stat_result)==0;

}

static void daemon_fsm_remove_replay_trigger_file_if_exists(void)
{
	const char *file_path = NULL;

	if(daemon_fsm_check_replay_trigger())
	{
		file_path = configuration_get_replay_level_trigger_path();
		if (unlink(file_path)!=0)
		{
			logger_log_error("Unable to remove the replay trigger file: %s",file_path);
		}
		helper_sync_parent_directory(file_path);
	}
}

static void daemon_fsm_create_level_change_complete_file(void)
{
	const char *file_path=NULL;
	FILE *fd=NULL;

	file_path=configuration_get_level_change_complete_file_path();
	fd=fopen(file_path,"w+");
	if (fd == NULL)
	{
		logger_log_error("Failed to create the file that marks level change completion: %s - %s",
				file_path, strerror(errno));
	}
	else
	{
		logger_log_debug("Created the file that marks the level change completion: %s ",
				file_path);

		fclose(fd);
		helper_sync_parent_directory(file_path);

		daemon_fsm_remove_replay_trigger_file_if_exists();
	}
}

static bool daemon_fsm_check_level_change_complete_file_exits(void)
{
	struct stat stat_result;
	return stat(configuration_get_level_change_complete_file_path(), &stat_result)==0;

}

static void daemon_fsm_remove_level_change_complete_file(void)
{
	if(daemon_fsm_check_level_change_complete_file_exits())
	{
		if (unlink(configuration_get_level_change_complete_file_path())!=0)
			logger_log_error("Unable to remove the file that marks level change completion: %s",
					configuration_get_level_change_complete_file_path());
		else
			logger_log_debug("Removed the file that marks level change completion: %s ",
					configuration_get_level_change_complete_file_path());

		helper_sync_parent_directory(configuration_get_level_change_complete_file_path());
	}
}
//-------------------------------------------------------------------------------------------------
